#include <wx/wx.h>
#include <wx/dc.h>

#include "horizon.h"

#include <iostream>

 using namespace std;

/*

These macro declarations describe how our window reacts to events. Various kind of events exits.
We have here only menu events to catch all our menu actions.

*/

BEGIN_EVENT_TABLE( horizonFrame, wxFrame )
	EVT_MENU( Menu_File_Quit, horizonFrame::OnQuit )
	EVT_MENU( Menu_File_About, horizonFrame::OnAbout )
END_EVENT_TABLE()


/*

This macro call link our application class to the entry point. It just calls some predefined wx methods implementing
the traditional C/C++ main(...) entry point.

*/
IMPLEMENT_APP(horizonapp)
	

/*
Automatically called after the application has been created. Everything is in order to create the whole GUI.
*/
bool 
horizonapp::OnInit()
{
// Create our main window
	horizonFrame *frame = new horizonFrame( wxT( "Horizon" ), wxPoint(50,50), wxSize(450,340) );
	frame->Show(TRUE);
	SetTopWindow(frame);
// We are happy, no error -> return true
	return TRUE;
} 

horizonFrame::horizonFrame( const wxString& title, const wxPoint& pos, const wxSize& size )
	: wxFrame((wxFrame *)NULL, -1, title, pos, size)
{
// The constructor of our main window
// Create the menu, and a status bar
	wxMenu *menuFile = new wxMenu;
	
	menuFile->Append( Menu_File_About, wxT( "&About..." ) );
	menuFile->AppendSeparator();
	menuFile->Append( Menu_File_Quit, wxT( "E&xit" ) );
	
	wxMenuBar *menuBar = new wxMenuBar;
	menuBar->Append( menuFile, wxT( "&File" ) );
	
	SetMenuBar( menuBar );
	
	CreateStatusBar();
	SetStatusText( wxT( "A nice (?!) artificial horizon !" ) );

	wxSizer *sizer=new wxBoxSizer(wxVERTICAL);  SetSizer(sizer);

	horizon=new ArtificialHorizon(this,wxPoint(0,0),wxSize(200,200));
	sizer->Add(horizon,1,wxEXPAND | wxALL , 1);
}

void 
horizonFrame::OnQuit( wxCommandEvent& WXUNUSED( event ) )
{
// Response to the menu "quit" event
// Do nothing but close the application
	Close(TRUE);
}

void 
horizonFrame::OnAbout( wxCommandEvent& WXUNUSED( event ) )
{
// Response to the menu "about" event
// show a small message
	wxMessageBox( wxT( "A sample software to show how to implement an artificial horizon" ),
			wxT( "Artificial horizon" ), wxOK | wxICON_INFORMATION, this );
}


BEGIN_EVENT_TABLE(ArtificialHorizon,wxWindow)
  EVT_PAINT(ArtificialHorizon::OnPaint)
  EVT_KEY_DOWN(ArtificialHorizon::OnKey)
END_EVENT_TABLE()


ArtificialHorizon::ArtificialHorizon(wxWindow *parent, const wxPoint& pos, const wxSize& size) :
	 wxWindow(parent, -1, pos, size , wxFULL_REPAINT_ON_RESIZE | wxWANTS_CHARS)
{
  // Initialize the horizon
	angle=height=0.;
	cx=cy=150;
	radius=135;
}

void ArtificialHorizon::OnKey(wxKeyEvent &ev)
{
  switch(ev.GetKeyCode()) {
    case WXK_LEFT : angle-=0.05; break;
    case WXK_RIGHT : angle+=0.05; break;
    case WXK_UP : height+=10; break;
    case WXK_DOWN : height-=10; break;
  }
  wxClientDC dc(this); draw(dc);
}

void ArtificialHorizon::OnPaint(wxPaintEvent&)
{
	wxPaintDC dc(this);
	draw(dc);
}

void ArtificialHorizon::transform(int x, int y, int &new_x, int &new_y,int trans_y)
{
// change the frame of (x,y), depending on the angle & height
// a translation followed by a rotation

// vertical translation
  int nx=x; int ny=trans_y ? y+height : y;
// rotation
  new_x=nx*cos(angle)-ny*sin(angle);
  new_y=nx*sin(angle)+ny*cos(angle);
}

void ArtificialHorizon::draw(wxDC &_dc)
{
  // draw the horizon

  // computes the horizon size depending on the window size
  wxSize size=GetVirtualSize();
  int wx=size.x; int wy=size.y;

  // to avoid flicker effects, draw everything on a bitmap, then draw the bitmap on the window
  wxBitmap draw_bitmap(wx,wy,-1);
  wxMemoryDC dc; dc.SelectObject(draw_bitmap);

  cx=wx/2; cy=wy/2;
  radius=(cx<cy ? cx : cy)*0.9;

  // draw the boundng box
  wxBrush grey_brush(wxColour(64,64,64),wxSOLID);
  dc.SetBrush(grey_brush);
  dc.DrawRectangle(0,0,wx,wy);

  // draw the external circle
  dc.SetBrush(*wxWHITE_BRUSH);
  dc.DrawCircle(cx,cy,radius);

  // create the clipping tool: it's nothing but this circle
  wxBitmap clip_bitmap(wx,wy,-1);
  wxMemoryDC mem_dc;
  mem_dc.SelectObject(clip_bitmap);
  // draw a balck rectangle, and a white circle on the bitmap
  mem_dc.SetBrush(*wxWHITE_BRUSH);
  mem_dc.DrawRectangle(0,0,wx,wy);
  mem_dc.SetBrush(*wxBLACK_BRUSH);
  mem_dc.DrawCircle(cx,cy,radius);

  wxRegion h_region(clip_bitmap,*wxWHITE);

  // clip all future drawings into the h_region, ie inside the center circle
  dc.DestroyClippingRegion();
  dc.SetClippingRegion(h_region);

  // test it, drawing 2 rectangles
  // dc.DrawRectangle(cx,cy,10,10);
  // dc.DrawRectangle(0,0,10,10);

  // Draw the two horizon regions
  wxPoint pts[10]; int i=0;

  // the lower black one
  transform(-radius-5,0,pts[i].x,pts[i].y); i++;
  transform(+radius+5,0,pts[i].x,pts[i].y); i++;
  transform(+radius+5,-2*wy,pts[i].x,pts[i].y); i++;
  transform(-radius-5,-2*wy,pts[i].x,pts[i].y); i++;

  for (i=0;i<4;i++) { pts[i].y=wy-(pts[i].y+cy); pts[i].x+=cx; }
  dc.SetBrush(*wxBLACK_BRUSH);
  dc.DrawPolygon(4,pts,0,0);

  // the upper black one
  i=0;
  transform(-radius-5,0,pts[i].x,pts[i].y); i++;
  transform(+radius+5,0,pts[i].x,pts[i].y); i++;
  transform(+radius+5,2*wy,pts[i].x,pts[i].y); i++;
  transform(-radius-5,2*wy,pts[i].x,pts[i].y); i++;

  for (i=0;i<4;i++) { pts[i].y=wy-(pts[i].y+cy); pts[i].x+=cx; }
  dc.SetBrush(wxBrush(wxColor(4,156,253),wxSOLID));
  dc.DrawPolygon(4,pts,0,0);

  // draw nice lines on the lower part
  i=0;
  transform(-radius+40,-radius+40,pts[i].x,pts[i].y); i++;
  transform(-10,-10,pts[i].x,pts[i].y); i++;
  for (i=0;i<2;i++) { pts[i].y=wy-(pts[i].y+cy); pts[i].x+=cx; }
  dc.SetPen(wxPen(*wxWHITE,2));
  dc.DrawLine(pts[0].x,pts[0].y,pts[1].x,pts[1].y);

  i=0;
  transform(radius-40,-radius+40,pts[i].x,pts[i].y); i++;
  transform(10,-10,pts[i].x,pts[i].y); i++;
  for (i=0;i<2;i++) { pts[i].y=wy-(pts[i].y+cy); pts[i].x+=cx; }
  dc.DrawLine(pts[0].x,pts[0].y,pts[1].x,pts[1].y);

  // draw some ticks on the upper part
  for (int j=-10;j<=10;j++) {
    i=0;
    transform(j%5==0 ? -10 : -5,j*10,pts[i].x,pts[i].y,0); i++;
    transform(j%5==0 ? 10 : 5,j*10,pts[i].x,pts[i].y,0); i++;
    for (i=0;i<2;i++) { pts[i].y=wy-(pts[i].y+cy); pts[i].x+=cx; }
    dc.DrawLine(pts[0].x,pts[0].y,pts[1].x,pts[1].y);
  }

  _dc.DrawBitmap(draw_bitmap,0,0);
  
}